/*
 * $QNXLicenseC:
 * Copyright 2010, QNX Software Systems.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You
 * may not reproduce, modify or distribute this software except in
 * compliance with the License. You may obtain a copy of the License
 * at: http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTIES OF ANY KIND, either express or implied.
 *
 * This file may contain contributions from others, either as
 * contributors under the License or as licensors under other terms.
 * Please review this entire file for other proprietary rights or license
 * notices, as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

/*
 * Synaptics Touchscreen driver
 *
 * Copyright (C) 2012-2018 Synaptics Incorporated. All rights reserved.
 *
 * The hardware access library is included from QNX software, and the
 * mtouch framework is also included from install/usr/input/. Both of
 * libraries are under Apache-2.0.
 *
 * Synaptics Touchscreen driver is interacted with QNX software system
 * as a touchscreen controller, for its licensing rule and related notices,
 * please scroll down the text to browse all of it.
 *
 * INFORMATION CONTAINED IN THIS DOCUMENT IS PROVIDED "AS-IS," AND SYNAPTICS
 * EXPRESSLY DISCLAIMS ALL EXPRESS AND IMPLIED WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
 * AND ANY WARRANTIES OF NON-INFRINGEMENT OF ANY INTELLECTUAL PROPERTY RIGHTS.
 * IN NO EVENT SHALL SYNAPTICS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, PUNITIVE, OR CONSEQUENTIAL DAMAGES ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OF THE INFORMATION CONTAINED IN THIS DOCUMENT, HOWEVER CAUSED
 * AND BASED ON ANY THEORY OF LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, AND EVEN IF SYNAPTICS WAS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE. IF A TRIBUNAL OF COMPETENT JURISDICTION DOES
 * NOT PERMIT THE DISCLAIMER OF DIRECT DAMAGES OR ANY OTHER DAMAGES, SYNAPTICS'
 * TOTAL CUMULATIVE LIABILITY TO ANY PARTY SHALL NOT EXCEED ONE HUNDRED U.S.
 * DOLLARS.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#ifndef UNITTEST
#include "i2c_client.h"
#include "dcmd_pm.h"
#include "utils.h"
#else
#include "err_mem_header.h"
#endif

#include "faceplate_rm.h"
#include "fidm_touch.h"
#include "syna_tcm2_mtouch.h"
#include "synaptics_touchcom_core_dev.h"
#include "synaptics_touchcom_func_base.h"
#include "synaptics_touchcom_func_touch.h"
#include "synaptics_touchcom_func_reflash.h"
#include "synaptics_touchcom_func_romboot.h"

#define XFER_ATTEMPTS 5

#define SYNA_MAX_TOUCH_POINTS 10

#define FLASH_ERASE_POLLING_DELAY (500)
#define FLASH_WRITE_POLLING_DELAY (50)

#define PLATFORM_DRIVER_NAME "syna_tcm_qnx"

/**
 * @brief USE_CUSTOM_TOUCH_REPORT_CONFIG
 *        Open if willing to set up the format of touch report.
 *        The custom_touch_format[] array in syna_tcm2.c can be used
 *        to describe the customized report format.
 */
/*#define USE_CUSTOM_TOUCH_REPORT_CONFIG*/

/**
 * @brief STARTUP_REFLASH
 *        Open if willing to do fw checking and update at startup.
 *        The path of firmware image will be defined by graphics.conf,
 *        so please ensure the target image is built-in properly.
 */
/* #define STARTUP_REFLASH */

#ifdef STARTUP_REFLASH
enum REFLASH_TYPE {
        IMAGE_REFLASH_LEGACY_DEV = 0,
        IMAGE_REFLASH_SINGLECHIP_TDDI,
        IMAGE_REFLASH_MULTIPLECHIP_TDDI,
        IHEX_REFLASH_SINGLECHIP_TDDI,
        IHEX_REFLASH_MULTIPLECHIP_TDDI,
};

#define REFLASH_TARGET_DUT IHEX_REFLASH_MULTIPLECHIP_TDDI


#define IS_IHEX_REFLASH(type) \
    ((type == IHEX_REFLASH_SINGLECHIP_TDDI) || \
    (type == IHEX_REFLASH_MULTIPLECHIP_TDDI))

#define IS_REFLASH_TO_MULTICHIP_TDDI(type) \
    ((type == IMAGE_REFLASH_MULTIPLECHIP_TDDI) || \
    (type == IHEX_REFLASH_MULTIPLECHIP_TDDI))

#endif

/**
 * @section: USE_CUSTOM_TOUCH_REPORT_CONFIG
 *           Open if willing to set up the format of touch report.
 *           The custom_touch_format[] array can be used to describe the
 *           customized report format.
 */
#ifdef USE_CUSTOM_TOUCH_REPORT_CONFIG
static unsigned char custom_touch_format[] = {
    /* entity code */                    /* bits */
    TOUCH_REPORT_TIMESTAMP,                 32,
    TOUCH_REPORT_NUM_OF_ACTIVE_OBJECTS,     8,
    TOUCH_REPORT_FOREACH_ACTIVE_OBJECT,
    TOUCH_REPORT_OBJECT_N_INDEX,            4,
    TOUCH_REPORT_OBJECT_N_CLASSIFICATION,   4,
    TOUCH_REPORT_OBJECT_N_X_POSITION,       12,
    TOUCH_REPORT_OBJECT_N_Y_POSITION,       12,
    TOUCH_REPORT_OBJECT_N_Z,                8,
    TOUCH_REPORT_OBJECT_N_X_WIDTH,          8,
    TOUCH_REPORT_OBJECT_N_Y_WIDTH,          8,
    TOUCH_REPORT_FOREACH_END,
    TOUCH_REPORT_KNOB_DATA,                 64,
    TOUCH_REPORT_END
};
#endif

/***************************************************************************************************
*   Static Functions' prototypes
***************************************************************************************************/
syna_static int mtouch_options(const char *option, const char *value, void *arg);
syna_static int mtouch_get_contact_id(void *packet, uint8_t digit_idx, uint32_t *contact_id, void *arg);
syna_static int mtouch_get_coords(void *packet, uint8_t digit_idx, int32_t *x, int32_t *y, void *arg);
syna_static int mtouch_is_contact_down(void *packet, uint8_t digit_idx, int *valid, void *arg);
syna_static int mtouch_tcm_set_up_app_fw(struct syna_tcm_dev *p_dev);
syna_static int syna_mtouch_spi_read(struct syna_hw_interface *hw_if,
        unsigned char *rd_data, unsigned int rd_len);
syna_static int syna_mtouch_i2c_read(struct syna_hw_interface *hw_if,
        unsigned char *rd_data, unsigned int rd_len);
syna_static int mtouch_bus_setup(struct syna_hw_interface *hw_if, unsigned char type);
syna_static int syna_mtouch_spi_write(struct syna_hw_interface *hw_if,
        unsigned char *wr_data, unsigned int wr_len);
/***************************************************************************************************/

/* Write critical logs into errmem */
void error_memory(const char * fmt, ...)
{
#define MAXLINE    1024   /* max text line length */

    va_list ap;
    char buf[MAXLINE] = {0};

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    vWritePrintfErrmem(buf);

}


static int syna_mtouch_connect_faceplate(struct syna_tcm_dev *p_dev);

/**
 * syna_mtouch_i2c_read()
 *
 * TouchCom over I2C uses the normal I2C addressing and transaction direction
 * mechanisms to select the device and retrieve the data.
 *
 * @param
 *    [ in] hw_if:   the handle of hw interface
 *    [out] rd_data: buffer for storing data retrieved from device
 *    [ in] rd_len: number of bytes retrieved from device
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
syna_static int syna_mtouch_i2c_read(struct syna_hw_interface *hw_if,
        unsigned char *rd_data, unsigned int rd_len)
{
    int retval;
    unsigned int attempt;
    struct syna_hw_bus_data *bus;
    // struct {
        // i2c_sendrecv_t hdr;
        // uint8_t buf[RD_CHUNK_SIZE];
    // } i2c_rd_data;

    _CHECK_POINTER(hw_if);

    _CHECK_POINTER(rd_data);

    bus = &hw_if->bdata_io;

    if (rd_len > RD_CHUNK_SIZE) {
        LOGE("Invalid input length %d (limit %d)", rd_len, RD_CHUNK_SIZE);
        error_memory("Synaptics_Touch: Invalid input length %d (limit %d)", rd_len, RD_CHUNK_SIZE);
        return -EINVAL;
    }
    if (bus->i2c_fd < 0) {
        LOGE("Invalid device descriptor");
        error_memory("Synaptics_Touch: Invalid device descriptor");
        return -EINVAL;
    }

    syna_pal_mutex_lock(&bus->io_mutex);

    // i2c_rd_data.hdr.slave.addr = bus->i2c_addr;
    // i2c_rd_data.hdr.slave.fmt = I2C_ADDRFMT_7BIT;
    // i2c_rd_data.hdr.send_len = 0;
    // i2c_rd_data.hdr.recv_len = rd_len;
    // i2c_rd_data.hdr.stop = 1;

    for (attempt = 0; attempt < XFER_ATTEMPTS; attempt++) {

        // retval = devctl(bus->i2c_fd,
                        // DCMD_I2C_SENDRECV,
                        // &i2c_rd_data,
                        // sizeof(i2c_rd_data),
                        // NULL);
        retval = i2c_read(bus->i2c_fd, rd_data, rd_len);
        if (rd_len == retval) {
            goto exit;
        }
        LOGE("Transfer attempt %d failed (error: %s)",
                attempt + 1, strerror(errno));

        if (attempt + 1 == XFER_ATTEMPTS) {
            retval = -EIO;
            goto exit;
        }

        //syna_pal_sleep_ms(20);
    }

exit:
    syna_pal_mutex_unlock(&bus->io_mutex);

    return retval;
}
/**
 * syna_mtouch_i2c_write()
 *
 * TouchCom over I2C uses the normal I2C addressing and transaction direction
 * mechanisms to select the device and send the data to the device.
 *
 * @param
 *    [ in] hw_if:   the handle of hw interface
 *    [ in] wr_data: written data
 *    [ in] wr_len: length of written data in bytes
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
static int syna_mtouch_i2c_write(struct syna_hw_interface *hw_if,
        unsigned char *wr_data, unsigned int wr_len)
{
    int retval;
    unsigned int attempt;
    struct syna_hw_bus_data *bus;
    // struct {
        // i2c_send_t hdr;
        // uint8_t buf[WR_CHUNK_SIZE];
    // } i2c_wr_data;

    _CHECK_POINTER(hw_if);

    _CHECK_POINTER(wr_data);

    bus = &hw_if->bdata_io;

    if (wr_len > WR_CHUNK_SIZE) {
        LOGE("Invalid input length %d (limit %d)", wr_len, WR_CHUNK_SIZE);
        error_memory("Synaptics_Touch: Invalid input length %d (limit %d)", wr_len, WR_CHUNK_SIZE);
        return -EINVAL;
    }
    if (bus->i2c_fd < 0) {
        LOGE("Invalid device descriptor");
        error_memory("Synaptics_Touch: Invalid device descriptor");
        return -EINVAL;
    }

    syna_pal_mutex_lock(&bus->io_mutex);

    // i2c_wr_data.hdr.slave.addr = bus->i2c_addr;
    // i2c_wr_data.hdr.slave.fmt = I2C_ADDRFMT_7BIT;
    // i2c_wr_data.hdr.len = wr_len;
    // i2c_wr_data.hdr.stop = 1;

    // memcpy(&i2c_wr_data.buf[0], wr_data, wr_len);

    for (attempt = 0; attempt < XFER_ATTEMPTS; attempt++) {

        // retval = devctl(bus->i2c_fd,
                    // DCMD_I2C_SEND,
                    // &i2c_wr_data,
                    // sizeof(i2c_wr_data),
                    // NULL);
        retval = i2c_write(bus->i2c_fd, wr_data, wr_len);
        if (wr_len == retval) {
            goto exit;
        }
        LOGE("Transfer attempt %d failed (error: %s)",
                attempt + 1, strerror(errno));

        if (attempt + 1 == XFER_ATTEMPTS) {
            retval = -EIO;
            goto exit;
        }


        //syna_pal_sleep_ms(20);
    }

exit:
    syna_pal_mutex_unlock(&bus->io_mutex);

    return retval;
}
/**
 * syna_mtouch_spi_read()
 *
 * TouchCom over SPI requires the host to assert the SSB signal to address
 * the device and retrieve the data.
 *
 * @param
 *    [ in] hw_if:   the handle of hw interface
 *    [out] rd_data: buffer for storing data retrieved from device
 *    [ in] rd_len: number of bytes retrieved from device
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
syna_static int syna_mtouch_spi_read(struct syna_hw_interface *hw_if,
        unsigned char *rd_data, unsigned int rd_len)
{
    int retval = 0;
    unsigned int attempt;
    struct syna_hw_bus_data *bus;
    // uint8_t tx_buf[WR_CHUNK_SIZE] = {0};
    // uint8_t rx_buf[WR_CHUNK_SIZE] = {0};

    _CHECK_POINTER(hw_if);

    _CHECK_POINTER(rd_data);

    bus = &hw_if->bdata_io;

    if (rd_len > RD_CHUNK_SIZE) {
        LOGE("Invalid input length %d (limit %d)", rd_len, RD_CHUNK_SIZE);
        error_memory("Synaptics_Touch: Invalid input length %d (limit %d)", rd_len, RD_CHUNK_SIZE);
        return -EINVAL;
    }
    if (bus->spi_fd < 0) {
        LOGE("Invalid device descriptor");
        error_memory("Synaptics_Touch: Invalid device descriptor");
        return -EINVAL;
    }

    syna_pal_mutex_lock(&bus->io_mutex);

    for (attempt = 0; attempt < XFER_ATTEMPTS; attempt++) {

        // retval = spi_xchange(bus->spi_fd,
                        // SPI_DEV_LOCK,
                        // tx_buf,
                        // rx_buf,
                        // (int)rd_len);
        if (rd_len == retval) {

            // memcpy(&rd_data[0], rx_buf, rd_len);

            goto exit;
        }
        LOGE("Transfer attempt %d failed (error: %s)",
                attempt + 1, strerror(errno));

        if (attempt + 1 == XFER_ATTEMPTS) {
            retval = -EIO;
            goto exit;
        }

       // syna_pal_sleep_ms(20);
    }

exit:
    syna_pal_mutex_unlock(&bus->io_mutex);

    return retval;
}
/**
 * syna_mtouch_spi_write()
 *
 * TouchCom over SPI requires the host to assert the SSB signal to address
 * the device and send the data to the device.
 *
 * @param
 *    [ in] hw_if:   the handle of hw interface
 *    [ in] wr_data: written data
 *    [ in] wr_len: length of written data in bytes
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
syna_static int syna_mtouch_spi_write(struct syna_hw_interface *hw_if,
        unsigned char *wr_data, unsigned int wr_len)
{
    int retval = 0;
    unsigned int attempt;
    struct syna_hw_bus_data *bus;
    // uint8_t tx_buf[WR_CHUNK_SIZE] = {0};
    // uint8_t rx_buf[WR_CHUNK_SIZE] = {0};

    _CHECK_POINTER(hw_if);

    _CHECK_POINTER(wr_data);

    bus = &hw_if->bdata_io;

    if (wr_len > WR_CHUNK_SIZE) {
        LOGE("Invalid input length %d (limit %d)", wr_len, WR_CHUNK_SIZE);
        error_memory("Synaptics_Touch: Invalid input length %d (limit %d)", wr_len, WR_CHUNK_SIZE);
        return -EINVAL;
    }
    if (bus->spi_fd < 0) {
        LOGE("Invalid device descriptor");
        error_memory("Synaptics_Touch: Invalid device descriptor");
        return -EINVAL;
    }

    syna_pal_mutex_lock(&bus->io_mutex);

    // memcpy(&tx_buf[0], &wr_data[0], wr_len);

    for (attempt = 0; attempt < XFER_ATTEMPTS; attempt++) {

        // retval = spi_xchange(bus->spi_fd,
                        // SPI_DEV_LOCK,
                        // tx_buf,
                        // rx_buf,
                        // (int)wr_len);
        if (wr_len == retval) {
            goto exit;
        }
        LOGE("Transfer attempt %d failed (error: %s)",
                attempt + 1, strerror(errno));

        if (attempt + 1 == XFER_ATTEMPTS) {
            retval = -EIO;
            goto exit;
        }

      //  syna_pal_sleep_ms(20);
    }

exit:
    syna_pal_mutex_unlock(&bus->io_mutex);

    return retval;
}

/**
 * mtouch_bus_setup()
 *
 * Configure and switch the bus connection
 *
 * The followings are the supported types.
 *      0:  BUS_NONE
 *      1:  BUS_I2C
 *      2:  BUS_SPI
 *
 * @param
 *    [ in] hw_if: the handle of hw interface
 *    [ in] type:  type of bus to connect
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_bus_setup(struct syna_hw_interface *hw_if, unsigned char type)
{
    int retval = 0;
    spi_cfg_t spi_cfg;
    // spi_devinfo_t spi_info;
    struct syna_hw_bus_data *bus;

    _CHECK_POINTER(hw_if);

    bus = &hw_if->bdata_io;

    if (BUS_TYPE_SPI == type) {

        spi_cfg.mode = (8 & SPI_MODE_CHAR_LEN_MASK) ;
        spi_cfg.clock_rate = bus->frequency_hz;

        if (bus->spi_mode == 3) {
            spi_cfg.mode |= SPI_MODE_CKPHASE_HALF;
        }
        else {
            /* this sources implements the mode.3 configuration only.
             * other mode configurations are not supported yet.
             */
            LOGW("Only mode.3 so far, request mode.%d not supported yet",
                bus->spi_mode);
            retval = _ENODEV;
            goto exit;
        }
        /* open spi device */
        // bus->spi_fd = spi_open(bus->spi_path);
        if (bus->spi_fd < 0) {
            LOGE("Fail to open SPI device %s", bus->spi_path);
            error_memory("Synaptics_Touch: Fail to open SPI device %s", bus->spi_path);
            retval = _ENODEV;
            goto exit;
        }
        /* set up spi config */
        // retval = spi_setcfg(bus->spi_fd,
                          // SPI_DEV_DEFAULT,
                          // &spi_cfg);
        if (EOK != retval) {
            LOGE("Fail to set up spi configuration");
            error_memory("Synaptics_Touch: Fail to set up spi configuration");
            goto exit;
        }
        /* check spi configuration */
        // retval = spi_getdevinfo(bus->spi_fd,
                            // SPI_DEV_DEFAULT,
                            // &spi_info);
        if (EOK != retval) {
            LOGE("Fail to get spi configuration");
            error_memory("Synaptics_Touch: Fail to get spi configuration");
            goto exit;
        }

        // LOGI("SPI Device: %s, name: %s, mode: %d",
            // bus->spi_path, spi_info.name,
            // spi_info.cfg.mode, spi_info.cfg.clock_rate);

        hw_if->ops_read_data = syna_mtouch_spi_read;
        hw_if->ops_write_data = syna_mtouch_spi_write;

        retval = bus->spi_fd;
    }
    /* initialize I2C interface */
    else if (BUS_TYPE_I2C == type) {

        /* open i2c device */
        // LOGI("I2c bus path %s", bus->i2c_path);
        // bus->i2c_fd = open(bus->i2c_path, O_RDWR);
        // if (bus->i2c_fd < 0) {
            // LOGE("Fail to open I2C device %s", bus->i2c_path);
            // retval = _ENODEV;
            // goto exit;
        // }

        // LOGI("I2c success fd: %d errno: %d", bus->i2c_fd, errno);
        // /* set up bus frequency */
        // retval = devctl(bus->i2c_fd,
                    // DCMD_I2C_SET_BUS_SPEED,
                    // &bus->frequency_hz,
                    // sizeof(bus->frequency_hz),
                    // NULL);
        // if (EOK != retval) {
            // LOGE("Failed to set i2c speed");
            // retval = _ENODEV;
            // goto exit;
        // }

        bus->i2c_fd = i2c_open(bus->i2c_path);
        if (bus->i2c_fd == -1) {
            LOGE("Fail to open I2C device %s", bus->i2c_path);
            error_memory("Synaptics_Touch: Fail to open I2C device %s", bus->i2c_path);
            retval = _ENODEV;
            goto exit;
        }

        if((i2c_set_slave_addr(bus->i2c_fd, bus->i2c_addr, I2C_ADDRFMT_7BIT)) == -1) {
            LOGE("Fail to open I2C device %s", bus->i2c_path);
            error_memory("Synaptics_Touch: Fail to open I2C device %s", bus->i2c_path);
            retval = _ENODEV;
            goto exit;
        }

        if((i2c_set_bus_speed(bus->i2c_fd, I2C_SPEED_HIGH, &bus->frequency_hz)) == -1)
            LOGD("Failed to set i2c speed, default speed will be set"); /* default speed is set,even on failure */

        LOGI("I2C registration success, Device: %s, address: 0x%02x",
            bus->i2c_path, bus->i2c_addr);

        hw_if->ops_read_data = syna_mtouch_i2c_read;
        hw_if->ops_write_data = syna_mtouch_i2c_write;

        retval = bus->i2c_fd;
    }
    else {
        LOGE("Unknown bus type setup, type = %d", type);
        error_memory("Synaptics_Touch: Unknown bus type setup, type = %d", type);
        retval = _EINVAL;
    }
exit:
    return retval;
}

/**
 * mtouch_bus_init()
 *
 * Assign all parameters at syna_hw_bus_data in defult
 *
 * @param
 *    [ in] hw_if: the handle of hw interface
 *
 * @return
 *    none
 */
static int mtouch_bus_init(struct syna_hw_interface *hw_if)
{
    struct syna_hw_bus_data *bus;

    _CHECK_POINTER(hw_if);

    bus = &hw_if->bdata_io;

    hw_if->pdev = NULL;

    bus->type = BUS_TYPE_NONE;
    bus->frequency_hz = 100000; // default 100k
    bus->i2c_fd = -1;
    bus->i2c_addr = 0x20;       // default at 0x20 slave address
    bus->spi_fd = -1;
    bus->spi_mode = 3;          // default at spi mode 3

    // create string buffers
    bus->i2c_path = (char *)syna_pal_mem_alloc(64, sizeof(char));
    if (!bus->i2c_path) {
        LOGE("Fail to create the i2c string buffer");
        error_memory("Synaptics_Touch: Fail to create the i2c string buffer");
        return _ENOMEM;
    }
    bus->spi_path = (char *)syna_pal_mem_alloc(64, sizeof(char));
    if (!bus->spi_path) {
        LOGE("Fail to create the spi string buffer");
        error_memory("Synaptics_Touch: Fail to create the spi string buffer");
        return _ENOMEM;
    }

    return 0;
}
/**
 * mtouch_irq_enable()
 *
 * Enable or disable the handling of interrupt
 *
 * @param
 *    [ in] hw_if: the handle of hw interface
 *    [ in] en:    '1' for enabling, and '0' for disabling
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static int mtouch_irq_enable(struct syna_hw_interface *hw_if, bool en)
{
    int retval;
    struct syna_hw_attn_data *attn;

    _CHECK_POINTER(hw_if);

    attn = &hw_if->bdata_attn;

    if (attn->irq_id == 0)
        return 0;

    syna_pal_mutex_lock(&attn->irq_en_mutex);

    /* enable the handling of interrupt */
    if (en) {
        if (attn->irq_enabled) {
            LOGI("Interrupt already enabled\n");
            retval = 0;
            goto exit;
        }

        retval = InterruptUnmask(attn->tp_intr, attn->irq_id);
        if (retval < 0) {
            LOGI("Fail to enable interrupt\n");
            goto exit;
        }

        attn->irq_enabled = true;

        LOGD("irq enabled\n", attn->irq_enabled);
    }
    /* disable the handling of interrupt */
    else {
        if (!attn->irq_enabled) {
            LOGI("Interrupt already disabled\n");
            retval = 0;
            goto exit;
        }

        retval = InterruptMask(attn->tp_intr, attn->irq_id);
        if (retval < 0) {
            LOGI("Fail to enable interrupt\n");
            goto exit;
        }

        attn->irq_enabled = false;

        LOGD("irq disabled\n", attn->irq_enabled);
    }

exit:
    syna_pal_mutex_unlock(&attn->irq_en_mutex);

    return retval;
}
/**
 * mtouch_power_on()
 *
 * Control the gpio to toggle a hardware reset
 * then power on the device
 *
 * @param
 *    [ in] hw_if: the handle of hw interface
 *    [ in] en:    '1' for powering on, and '0' for powering off
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_power_on(struct syna_hw_interface *hw_if, bool en)
{
    /* if power gpio is connected,
     * the power sequence can be implemented her
     */
    return 0;
}

syna_static int mtouch_reset(struct syna_hw_interface *hw_if)
{
    uint8_t  ret, len = 2;
    uint8_t buf[2] = {0};
    int fd;
    int Ctrl_handle, RetVal = -1;

    Ctrl_handle = displaybinder_ctrl_open(hw_if->bdata_rst.dev_ctrl_path);  // dev_control_path -> /dev/vcd/display-binder/control

    if(Ctrl_handle < 0)
    {
        LOGE("Error to open display binder control path Retval: %d", Ctrl_handle);
    } else {
       RetVal = displaybinder_ctrl_set(Ctrl_handle, "touch-reset","one-time");
       if(RetVal < 0)
       {
           LOGE("Failed to reset the touch controller through FIDM API retval: %d, errno:%d", RetVal, errno);
           displaybinder_ctrl_close(Ctrl_handle);
       } else {
             LOGI("executed touch reset successfully through FIDM API\n");
             displaybinder_ctrl_close(Ctrl_handle);
             return RetVal;
       }
    }
    if((Ctrl_handle < 0) || (RetVal < 0))
    {
        fd = i2c_open(hw_if->bdata_io.i2c_path);

        if (fd == -1)
            return -ENODEV;

        if((i2c_set_slave_addr(fd, 0x12, I2C_ADDRFMT_7BIT)) == -1) {
            LOGE("Fail to set mcu slave address %s", hw_if->bdata_io.i2c_path);
            error_memory("Synaptics_Touch: Fail to set mcu slave address %s", hw_if->bdata_io.i2c_path);
            i2c_close(fd);
            return -1;
        }

        if((i2c_set_bus_speed(fd, I2C_SPEED_HIGH, &hw_if->bdata_io.frequency_hz)) == -1) {
            LOGD("Failed to set i2c speed for mcu, default speed will be set"); /* default speed is set,even on failure */
        }

        buf[0] = 0x31;
        buf[1] = 0x01;

        ret = i2c_write(fd, buf, len);
        i2c_close(fd);

        if(ret == len) {
            return 0;
        }
    }
    return RetVal;
}

/**
 * syna_hw_interface
 *
 * Provide the hardware specific interface on the target platform
 */
static struct syna_hw_interface mtouch_hw_if = {
    .bdata_io = {
        .type = BUS_TYPE_I2C,
        .rd_chunk_size = RD_CHUNK_SIZE,
        .wr_chunk_size = WR_CHUNK_SIZE,
    },
    .bdata_attn = {
        .irq_enabled = false,
        .irq_on_state = 0,
        .tp_intr = 0xFFFF,
    },
    .bdata_rst = {
        .reset_on_state = 0,
        .reset_delay_ms = 200,
        .reset_active_ms = 20,
    },
    .bdata_pwr = {
        .power_on_state = 1,
        .power_on_delay_ms = 200,
    },
    .ops_power_on = mtouch_power_on,
    .ops_enable_irq = mtouch_irq_enable,
    .ops_bus_setup = mtouch_bus_setup,
    .ops_hw_reset = mtouch_reset,
    /* data reads / writes is configurable at ops_bus_setup() */
    .ops_read_data = NULL,
    .ops_write_data = NULL,
};

int syna_mtouch_register_pm(struct syna_tcm_dev *dev)
{
    struct pm_register_s pm_reg;
    int ret;

    dev->pm_fd = open(dev->pm_dev, O_RDWR  | O_CLOEXEC);
    if (dev->pm_fd  == -1) {
        LOGE("dev pm open() failed %d", strerror(errno));
        error_memory("Synaptics_Touch: dev pm open() failed %s", strerror(errno));
        goto fail;
    }

    memset(&pm_reg, 0x0, sizeof(struct pm_register_s));
    INIT_PM_REGISTER_STRUCT(&pm_reg);

    strlcpy(pm_reg.name, MTOUCH_DEV, sizeof(pm_reg.name));
    pm_reg.pulse_codes[PM_STATE_PREPARE] = SYNAPTICS_PM_PREPARE_PULSE;
    pm_reg.pulse_codes[PM_STATE_SUSPEND] = SYNAPTICS_PM_SUSPEND_PULSE;
    pm_reg.pulse_codes[PM_STATE_RESUME] = SYNAPTICS_PM_RESUME_PULSE;
    pm_reg.pulse_codes[PM_STATE_COMPLETE] = -1;
    pm_reg.priority = PM_PRIO_LEVEL_0;
    pm_reg.flags = 0;
    pm_reg.chid = dev->thread_chid;

    /* Register Cypresss with Power Manager */
    ret = devctl(dev->pm_fd, DCMD_PM_REGISTER, &pm_reg, sizeof(struct pm_register_s), NULL);
    if (ret != EOK) {
        LOGE("devctl() failed: %s\n", strerror(errno));
        error_memory("Synaptics_Touch: devctl() failed: %s\n", strerror(errno));
        goto fail_pm;
    }
    LOGI("/dev/pm registration successful %d\n", ret);

    /* Success */
    return 0;

fail_pm:
    close(dev->pm_fd);
fail:
    return -1;
}

void *syna_ext_msg_handler(void *arg)
{
    struct syna_tcm_dev *p_dev = arg;
    fidm_touch_msg_u msg;
    int rcvid;
    int ret;

    while (1) {
        rcvid = MsgReceive(p_dev->msg_thread->fidm_attach->chid, &msg, sizeof(msg), NULL);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }
            LOGE("External thread MsgReceive failed: %s %d", strerror(errno), rcvid);
            goto fail;
        } else if (rcvid != 0) {
            /* if synaptics is in suspend/prepare state, return EAGAIN to try after sometime
             * process for FIDM_TOUCH_GET_GROUPID irrespective of pm state
             */
            if ((p_dev->syn_pm_state != PM_STATE_RESUME) && (msg.type != FIDM_TOUCH_GET_GROUPID)) {
                MsgError(rcvid, -EAGAIN);
                continue;
            }
            /* msg received */
            switch(msg.type) {
                case FIDM_TOUCH_GET_RTD_DATA:
                    if (p_dev->verbose > 5)
                        LOGI("Received FIDM_TD_MSG_GET_RTD_DATA");

                    if (p_dev->touch_rtd_readflag == 1) {
                        ret = MsgReply(rcvid, 0, p_dev->touch_rtd, p_dev->touch_rtd_len);
                        if (ret == EOK) {
                            p_dev->touch_rtd_readflag = 0;
                        }
                        else {
                            MsgError(rcvid, -errno);
                        }
                        if (p_dev->verbose > 5)
                            LOGI("Msg replied with touch rtd data: %d", errno);
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;
                case FIDM_TOUCH_GET_KNOB_RTD_DATA:
                    if (p_dev->verbose > 5)
                        LOGI("Received FIDM_TOUCH_GET_KNOB_RTD_DATA");
                    if (p_dev->knob_rtd_readflag == 1) {
                        ret = MsgReply(rcvid, 0, p_dev->knob_rtd, p_dev->knob_rtd_len);
                        if (ret == EOK) {
                            p_dev->knob_rtd_readflag = 0;
                        }
                        else {
                            MsgError(rcvid, -errno);
                        }
                        if (p_dev->verbose > 5)
                            LOGI("Msg replied with knob rtd data: %d", errno);
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;
                 case FIDM_TOUCH_GET_GROUPID:
                     LOGI("Received FIDM_TOUCH_GET_GROUPID");
                     if(p_dev->display_group_id < 0) {
                         ret = syna_tcm_get_display_grpid(p_dev);
                         if(ret != 0)
                         {
                            LOGE("failed to fetch group id, ret : %d", ret);
                            error_memory("Synaptics_Touch: failed to fetch group id, ret : %d", ret);
                            MsgError(rcvid, -errno);
                         } else {
                            ret = MsgReply(rcvid, 0, &p_dev->display_group_id, sizeof(p_dev->display_group_id));
                            if (ret != EOK) {
                                 MsgError(rcvid, -errno);
                            }
                         }
                     } else {
                         ret = MsgReply(rcvid, 0, &p_dev->display_group_id, sizeof(p_dev->display_group_id));
                         LOGI("GROUPID %d ", p_dev->display_group_id);
                         if(ret != EOK) {
                            MsgError(rcvid, -errno);
                         }
                     }
                     break;

                default:
                    if (p_dev->verbose > 5)
                    {
                        LOGE("%s: Invalid msg id received", __FUNCTION__);
                        error_memory("Synaptics_Touch: %s: Invalid msg id received", __FUNCTION__);
                    }
                    MsgError(rcvid, -ENOTSUP);
                    break;
            }
        }
    }
fail:
    LOGE("MsgReceive failed for external message handler");
    return NULL;
}

void syna_tcm_dump_rtd_record(struct tcm_dev *tcm_dev, uint8_t len)
{
    char *buf;
    int i;

    if (!tcm_dev) {
        LOGE("Invalid tcm device handle\n");
        error_memory("Synaptics_Touch: Invalid tcm device handle\n");
        return;
    }

    buf = (char *) malloc (len);
    if (buf == NULL) {
        LOGE ("Failed to allocate buffer space to dump rtd record");
        return;
    }
    sprintf (buf, "%x ", tcm_dev->report_buf.buf[0]);
    for (i = 1; i < len ; i++) {
        sprintf ((buf + strlen(buf)), "%x ", tcm_dev->report_buf.buf[i]);
    }

    LOGD("rtd data = %s", buf);
    free(buf);

}

int syna_tcm_get_display_grpid(struct syna_tcm_dev *p_dev)
{
    int handle, ret;
    char *reply;
    reply = NULL;

    if(!p_dev->dev_status_path)
    {
        LOGE("Not a valid status path %s", p_dev->dev_status_path);
        error_memory("Synaptics_Touch: Not a valid status path %s", p_dev->dev_status_path);
        return -EINVAL;
    }
    handle = displaybinder_status_open(p_dev->dev_status_path);
    if(handle < 0)
    {
        LOGE("Failed to open display binder status path: %s, handle:%d", p_dev->dev_status_path, handle);
        error_memory("Synaptics_Touch: Failed to open display binder status path: %s, handle:%d", p_dev->dev_status_path, handle);
        return handle;
    }
    LOGD("display binder open success(reval: %d) for dev status path %s", handle, p_dev->dev_status_path);
    ret = displaybinder_status_get(handle,"display-group-id",&reply);
    displaybinder_status_close(handle);
    if (ret) {
       LOGE("failed to fetch group id: %d", ret);
       error_memory("Synaptics_Touch: failed to fetch group id: %d", ret);
    } else {
        p_dev->display_group_id = atoi(reply);
        LOGI("Display Group ID is fetched successfully: %d", p_dev->display_group_id);
    }

    if(reply)
       free(reply);
    return ret;
}

int syna_tcm_process_touch_rtd(struct syna_tcm_dev *p_dev)
{
    struct tcm_dev *tcm_dev;
    int ret;

    _CHECK_POINTER(p_dev);

    tcm_dev = p_dev->tcm_dev;

    if (tcm_dev->report_buf.data_length == MAX_RTD_DATA_LENGTH -1){
        p_dev->touch_rtd_len = MAX_RTD_DATA_LENGTH;
    }
    else {
        LOGE("Invalid touch rtd length %d\n", tcm_dev->report_buf.data_length);
        error_memory("Synaptics_Touch: Invalid touch rtd length %d\n", tcm_dev->report_buf.data_length);
        return -1;
    }

    if(p_dev->display_group_id < 0)
    {
        ret = syna_tcm_get_display_grpid(p_dev);
        if(ret != 0)
        {
           LOGE("failed to fetch group id: %d again", p_dev->display_group_id);
           error_memory("Synaptics_Touch: failed to fetch group id: %d again", p_dev->display_group_id);
           return -1;
        }
    }

    memcpy(p_dev->touch_rtd, tcm_dev->report_buf.buf, tcm_dev->report_buf.data_length);
    p_dev->touch_rtd[MAX_RTD_DATA_LENGTH-1] = p_dev->display_group_id;

    if (p_dev->verbose > 6) {
        syna_tcm_dump_rtd_record(tcm_dev, MAX_RTD_DATA_LENGTH);
    }

    return 0;
}

int syna_tcm_process_knob_rtd(struct syna_tcm_dev *p_dev)
{
    struct tcm_dev *tcm_dev;

    _CHECK_POINTER(p_dev);

    tcm_dev = p_dev->tcm_dev;

    if (tcm_dev->report_buf.data_length == MAX_KNOB_RTD_DATA_LENGTH){
        p_dev->knob_rtd_len = MAX_KNOB_RTD_DATA_LENGTH;
    }
    else {
         LOGE("Invalid knob rtd length %d\n", tcm_dev->report_buf.data_length);
         error_memory("Synaptics_Touch: Invalid knob rtd length %d\n", tcm_dev->report_buf.data_length);
         return -1;
    }

    memcpy(p_dev->knob_rtd, tcm_dev->report_buf.buf, MAX_KNOB_RTD_DATA_LENGTH);

    if (p_dev->verbose > 6) {
        syna_tcm_dump_rtd_record(tcm_dev, tcm_dev->report_buf.data_length);
    }

    return 0;
}

/**
 * syna_tcm_send_knob_data()
 *
 * Function to send the knob data to faceplate resource manager
 *
 * @param
 *    [ in] p_dev: device handle
 *
 * @return
 *    on success, 0 or positive value; otherwise, negative value on error.
 */
int syna_tcm_send_knob_data(struct syna_tcm_dev *p_dev)
{
    int ret;
    faceplate_msg msg;
    if (p_dev->faceplate_coid == -1)
    {
        LOGI("syna_tcm_send_knob_data: Trying to connect to faceplate");
        if ((ret = syna_mtouch_connect_faceplate(p_dev)) < 0)
        {
            LOGE("syna_tcm_send_knob_data: FAILED to connect faceplate");
            error_memory("Synaptics_Touch: syna_tcm_send_knob_data: FAILED to connect faceplate");
            return -EAGAIN;
        }
    }
    msg.message.command = FACEPLATE_KNOB_SUPPORT;
    msg.message.buff = p_dev->tcm_dev->knob_data;
    ret = MsgSend(p_dev->faceplate_coid, &msg, sizeof(msg), NULL, 0);
    if (ret != EOK)
    {
        LOGE("syna_tcm_send_knob_data: FAILED to connect faceplate");
        error_memory("Synaptics_Touch: syna_tcm_send_knob_data: FAILED to connect faceplate");
    }

    return ret;
}


/**
 * mtouch_interrupt_handling_thread()
 *
 * Implement the interrupt handling routine, which is created by
 * pthread_create() at mtouch_driver_init().
 *
 * This function is designed as a message-driven thread, and runs
 * until driver status is cleared.
 *
 * @param
 *    [ in] args: the mtouch device instance data, syna_dev_t
 *
 * @return
 *    none
 */
static void *mtouch_interrupt_handling_thread(void *args)
{
    int retval;
    unsigned char event_code = 0;
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)args;
    struct syna_hw_attn_data *attn;
    /* the input/output vector */
    iov_t iov;
    int    rcvid;
    struct pm_ack_s ack = { .rc = 0 };
    /* the _pulse structure describes a pulse, a fixed-size,
     * nonblocking message that carries a small payload
     */
    struct _pulse pulse;
    /* structure that describes an event */
    struct sigevent ev;


    /* prepare for the simple messages passing
     * fill in the fields of an iov_t structure
     */
    SETIOV (&iov, &pulse, sizeof(pulse));

    /* enable IO capability
     * lock the process's memory and request I/O privileges
     * let the thread execute the in, ins, out, outs, cli,
     * and sti I/O opcodes
     */
    if (ThreadCtl(_NTO_TCTL_IO, 0) == -1) {
        LOGE("Fail to config ThreadCtl");
        error_memory("Synaptics_Touch: Fail to config ThreadCtl");
        return 0;
    }

    attn = &p_dev->hw_if->bdata_attn;

    /* keep waiting and receiving ATTN signal until
     * driver status is cleared.
     */
    while (p_dev->status == STATUS_INIT) {

        /* the MsgReceivev kernel calls wait for a message to
         * arrive on the channel identified by chid, and place
         * the received data in the array of buffers, iov
         */
        if ((rcvid = MsgReceivev(p_dev->thread_chid, &iov, 1, NULL)) == -1) {
            /* if the child channel doesn't exist. */
            if (errno == ESRCH) {
                LOGE("Fail on MsgReceivev");
                error_memory("Synaptics_Touch: Fail on MsgReceivev");
                pthread_exit (NULL);
            }
            continue;
        }

        switch (pulse.code) {
            case PULSE_CODE:
            /* once receiving an ATTN, to read events generated by device and
             * retrieve all enqueued messages until ATTN is no longer asserted.
             */
            syna_pal_mutex_lock(&p_dev->tp_event_mutex);

            if (p_dev->verbose > 5)
                LOGD("Interrupt received");

            // /* read in the message packet generated by the device.
             // *
             // * raw report data will be stored at p_dev->event_data buffer;
             // * while the p_dev->tp_data contains the parsed touch report.
             // */
            /* Process get event data only on RESUME*/
            //if(p_dev->syn_pm_state == PM_STATE_RESUME)
            //{

            if (p_dev->tcm_dev->read_message != NULL) {
                retval = syna_tcm_get_event_data(p_dev->tcm_dev,
                    &event_code,
                    &p_dev->event_data,
                    p_dev->tp_data);
                if (retval < 0) {
                LOGE("Fail to get event data\n");
                error_memory("Synaptics_Touch: Fail to get event data\n");
                event_code = 0;
                }
            } else {
                  if (p_dev->verbose > 5)
                      LOGD("tcm_dev->read_message: value is null, so skiping syna_tcm_get_event_data function call");
            }
            //}
            //else
            //{
               //event_code = 0;
            //}

            SIGEV_NONE_INIT(&ev);
            SIGEV_MAKE_OVERDRIVE(&ev);
            MsgDeliverEvent(0, &ev);

            switch (event_code) {
            case REPORT_TOUCH_OLD:
            case REPORT_TOUCH:
                /* report input event only when receiving a touch report
                 *
                 * the relevant information of touch report will be used to create an mtouch_event_t,
                 * so that the Input Events framework can continue to process it and pass it to Screen.
                 */
                //if (p_dev->verbose > 5)
                //    LOGI("touch data sent\n");
                LOGI("Finger: %d, x: %d, y:  %d, status: %s\n", p_dev->tp_data->idx,
                         p_dev->tp_data->object_data[p_dev->tp_data->idx].x_pos,
                         p_dev->tp_data->object_data[p_dev->tp_data->idx].y_pos,
                         (p_dev->tp_data->object_data[p_dev->tp_data->idx].status ? "Pressed" : "Release"));

                if (p_dev->inputevents_hdlr != NULL) {
                    mtouch_driver_process_packet(p_dev->inputevents_hdlr,
                                            p_dev,
                                            p_dev,
                                            MTOUCH_PARSER_FLAG_NONE);
                }
                else {
                    if (p_dev->verbose > 5)
                         LOGI("mtouch_interrupt_handling_thread: p_dev->inputevents_hdlr is null so discarding pulse\n");
                }
                break;

            case REPORT_KNOB:
                if (p_dev->verbose > 5) {
                    LOGI("mtouch_interrupt_handling_thread: Interrupt received for knob");
                }
                retval = syna_tcm_send_knob_data(p_dev);
                if(retval != 0)
                {
                    LOGE("mtouch_interrupt_handling_thread: Failed to send the knob data to faceplate. retval = %d",retval);
                    error_memory("Synaptics_Touch: mtouch_interrupt_handling_thread: Failed to send the knob data to faceplate. retval = %d",retval);
                }
                else
                {
                    LOGI("mtouch_interrupt_handling_thread: Knob data sent successfully. Data = %d",p_dev->tcm_dev->knob_data);
                }
                break;

            case REPORT_TOUCH_DIAG:
                 retval = syna_tcm_process_touch_rtd(p_dev);
                 if (retval != 0) {
                     LOGE("Fail to get touch rtd data %d\n", retval);
                     error_memory("Synaptics_touch: Fail to get touch rtd data %d\n", retval);
                 } else {
                      p_dev->touch_rtd_readflag = 1;
                 }
                break;
            case REPORT_KNOB_DIAG:
                 retval = syna_tcm_process_knob_rtd(p_dev);
                 if (retval != 0) {
                     LOGE("Fail to get knob rtd data %d\n", retval);
                     error_memory("Synaptics_touch: Fail to get knob rtd data %d\n", retval);
                 } else {
                      p_dev->knob_rtd_readflag = 1;
                 }
                break;

            default:
                break;
            }

            syna_pal_mutex_unlock(&p_dev->tp_event_mutex);

            /* after the interrupt-handling thread has dealt with the event,
             * this must be called InterruptUnmask to re-enable the interrupt.
             */
            InterruptUnmask(attn->tp_intr, attn->irq_id);
        break;
        case SYNAPTICS_PM_PREPARE_PULSE:
                LOGI("Received SYNAPTICS_PM_PREPARE_PULSE");
                if(p_dev->syn_pm_state == PM_STATE_SUSPEND)
                {
                    LOGI("Ignoring the SYNAPTICS_PM_PREPARE_PULSE since mtouch driver is already in PREPARE state");

                    /* Send ACK to Power Manager */
                    ack.rc = 0;
                } else {
                    /* prepare synaptics suspend */
                    // do nothing

                    if (ack.rc) {
                        LOGE("Failed to handle SYNAPTICS_PM_PREPARE_PULSE");
                        error_memory("Synaptics_Touch: Failed to handle SYNAPTICS_PM_PREPARE_PULSE");
                    } else {
                        /* Update synaptics PM state */
                        p_dev->syn_pm_state = PM_STATE_PREPARE;
                    }
                }
                /* send ACK to power manager */
                ack.state = PM_STATE_PREPARE;
                retval = devctl(p_dev->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                if (retval != EOK) {
                    LOGE("devctl(DCMD_PM_ACK) for SYNAPTICS_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                    error_memory("Synaptics_Touch: devctl(DCMD_PM_ACK) for SYNAPTICS_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                }
        break;
        case SYNAPTICS_PM_SUSPEND_PULSE:
            LOGI("Received SYNAPTICS_PM_SUSPEND_PULSE");
            if(p_dev->syn_pm_state == PM_STATE_SUSPEND)
            {
                LOGI("Ignoring the SYNAPTICS_PM_SUSPEND_PULSE since mtouch driver is already in SUSPEND state");

                /* Send ACK to Power Manager */
                ack.rc = 0;
            } else {
                /* process mxt suspend */
                ack.rc = syna_mtouch_suspend(p_dev);

                if (ack.rc) {
                    LOGE("Failed to handle SYNAPTICS_PM_SUSPEND_PULSE");
                    error_memory("Synaptics_Touch: Failed to handle SYNAPTICS_PM_SUSPEND_PULSE");
                } else {
                   /* Update MXT PM state */
                   p_dev->syn_pm_state = PM_STATE_SUSPEND;
                }
            }
            /* Send ACK to Power Manager */
            ack.state = PM_STATE_SUSPEND;
            retval = devctl(p_dev->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
            if (retval != EOK) {
                LOGE("devctl(DCMD_PM_ACK) for SYNAPTICS_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
                error_memory("Synaptics_Touch: devctl(DCMD_PM_ACK) for SYNAPTICS_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
            }
        break;
        case SYNAPTICS_PM_RESUME_PULSE:
            LOGI("Received SYNAPTICS_PM_RESUME_PULSE");
            /* process synaptics resume */
             ack.state = PM_STATE_RESUME;
            retval = devctl(p_dev->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
            if (retval != EOK) {
                LOGE("devctl(DCMD_PM_ACK) for SYNAPTICS_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
                error_memory("Synaptics_Touch: devctl(DCMD_PM_ACK) for SYNAPTICS_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
            } else { 
               LOGI("Sent the ack to power manager about the RESUME Pulse received");
            }
            if(p_dev->syn_pm_state == PM_STATE_RESUME)
            {
                LOGI("Ignoring the SYNAPTICS_PM_RESUME_PULSE since mtouch driver is already in RESUME state");
                /* Send ACK to Power Manager */
                ack.rc = 0;
            }
            else {
                ack.rc = syna_mtouch_resume(p_dev);
                /* Send ACK to Power Manager */
                if (ack.rc) {
                LOGE("Failed to handle MXT_PM_RESUME_PULSE");
                error_memory("Synaptics_Touch: Failed to handle MXT_PM_RESUME_PULSE");
                } else {
                    /* Update synaptics PM state */
                    p_dev->syn_pm_state = PM_STATE_RESUME;
                }
            }
        break;
        default:
            LOGE("Unknown pulse code %x", pulse.code);
            error_memory("Synaptics_Touch: Unknown pulse code %x", pulse.code);
            if (rcvid) {
                MsgReplyv(rcvid, ENOTSUP, &iov, 1);
            }
        break;
        }
    }

    LOGE("Interrupt handling thread terminated");
    error_memory("Synaptics_Touch: Interrupt handling thread terminated");

    return 0;
}

void syna_release_touch(struct syna_tcm_dev *p_dev)
{
    struct tcm_touch_data_blob *tp_data;
    int idx;
    int pending_release = 0;

    if (!p_dev || !p_dev->tp_data)
        return;

    tp_data = p_dev->tp_data;

    for (idx = 0; idx < SYNA_MAX_TOUCH_POINTS; idx++)
    {
        // Check if the touch point is active
        if (tp_data->object_data[idx].status > 0)
        {
            pending_release++;
            // Reset the touch status
            tp_data->object_data[idx].status = 0;
            // Update the previous object status
            p_dev->prev_obj_status[idx] = 0;
            LOGI("Releasing touch [%d] Coordinate X: %d Y: %d Status: %s\n",
                 idx,
                 tp_data->object_data[idx].x_pos,
                 tp_data->object_data[idx].y_pos,
                 (tp_data->object_data[idx].status ? "Pressed" : "Released"));
        }
    }
    // If there is any pending release, process the packet
    if (pending_release > 0) {
        LOGI("release_pending_touches entered - Pending Release - %d\n", pending_release);
        mtouch_driver_process_packet(p_dev->inputevents_hdlr, p_dev, p_dev, MTOUCH_PARSER_FLAG_NONE);
        LOGI("Released all %d touch pointers by process packet\n", pending_release);
    }
}

/**
 * mtouch_is_contact_down()
 *
 * Retrieve the touch status for the specified digit of a touch-related event
 *
 * This function is called for each of the touchpoints.
 * The maximum number of touchpoints is max_touchpoints, as specified in mtouch_driver_params_t
 *
 * @param
 *    [ in] packet:    data packet that contains information on the touch-related event
 *    [ in] digit_idx: digit (finger) index that the Input Events library is requesting, 0 ~ (max_touchpoints-1)
 *                     this function will be invoked in (max_touchpoints-1) times
 *    [out] valid:     touch status (e.g., 1 = Down and 0 = Up) returned
 *    [ in] arg:       user information
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_is_contact_down(void *packet, uint8_t digit_idx, int *valid, void *arg)
{
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)packet;
    struct tcm_touch_data_blob *tp_data;

    _CHECK_POINTER(p_dev);

    tp_data = p_dev->tp_data;

    if ((p_dev->prev_obj_status[digit_idx] == 0) && tp_data->object_data[digit_idx].status == 0)
        *valid = 0;

    if (tp_data->object_data[digit_idx].status > 0)
        *valid = 1;
    else
        *valid = 0;

    if (p_dev->prev_obj_status[digit_idx] != tp_data->object_data[digit_idx].status) {
        p_dev->prev_obj_status[digit_idx] = tp_data->object_data[digit_idx].status;

        LOGI("Touch [%d] : %s\n", digit_idx, (*valid)? "TOUCH_DOWN":"TOUCH_UP");
    }

    return EOK;
}
/**
 * mtouch_get_contact_id()
 *
 * Retrieve the contact ID for the specified digit of a touch-related event
 *
 * This function is called for each of the touchpoints.
 * The maximum number of touchpoints is max_touchpoints, as specified in mtouch_driver_params_t
 *
 * @param
 *    [ in] packet:     data packet that contains information on the touch-related event
 *    [ in] digit_idx:  digit (finger) index that the Input Events library is requesting, 0 ~ (max_touchpoints-1)
 *                      this function will be invoked in (max_touchpoints-1) times
 *    [out] contact_id: contact ID returned
 *    [ in] arg:        user information
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_get_contact_id(void *packet, uint8_t digit_idx, uint32_t *contact_id, void *arg)
{
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)packet;

    *contact_id = digit_idx;
    if(p_dev->verbose > 5)
        LOGI("Touch [%d] : id = %-2d\n", digit_idx, *contact_id);

    return EOK;
}

/**
 * mtouch_get_coords()
 *
 * Retrieve the coordinates for the specified digit of a touch-related event
 *
 * This function is called for each of the touchpoints.
 * The maximum number of touchpoints is max_touchpoints, as specified in mtouch_driver_params_t
 *
 * @param
 *    [ in] packet:     data packet that contains information on the touch-related event
 *    [ in] digit_idx:  digit (finger) index that the Input Events library is requesting, 0 ~ (max_touchpoints-1)
 *                      this function will be invoked in (max_touchpoints-1) times
 *    [out] x:          x position of the touched finger 'digit_idx'
 *    [out] y:          y position of the touched finger 'digit_idx'
 *    [ in] arg:        user information
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_get_coords(void *packet, uint8_t digit_idx, int32_t *x, int32_t *y, void *arg)
{
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)packet;
    struct tcm_touch_data_blob *tp_data;

    _CHECK_POINTER(p_dev);

    tp_data = p_dev->tp_data;

    _CHECK_POINTER(tp_data);

    tp_data = p_dev->tp_data;

    *x = tp_data->object_data[digit_idx].x_pos;
    *y = tp_data->object_data[digit_idx].y_pos;

    LOGI("Touch [%d] : (x , y) = (%-4d, %-4d)\n", digit_idx, *x, *y);

    return EOK;
}


/**
 * mtouch_attach_device()
 *
 * Attach the driver to the Input Event framework, libinputevents
 *
 * Assign the callback functions appropriately by using the mtouch_driver_funcs_t,
 * and specify the capabilities that are supported in mtouch_driver_params_t
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static int mtouch_attach_device(struct syna_tcm_dev *p_dev)
{
    _CHECK_POINTER(p_dev);
    _CHECK_POINTER(p_dev->tcm_dev);

    /* assign the supported callbacks */
    mtouch_driver_funcs_t funcs = {
        .get_contact_id = mtouch_get_contact_id,
        .is_contact_down = mtouch_is_contact_down,
        .get_coords = mtouch_get_coords,
        .get_down_count = NULL,
        .get_touch_width = NULL,
        .get_touch_height = NULL,
        .get_touch_orientation = NULL,
        .get_touch_pressure = NULL,
        .get_seq_id = NULL,
        .set_event_rate = NULL,
        .get_contact_type = NULL,
        .get_select = NULL
    };

    /* specify the capabilities */
    mtouch_driver_params_t params = {
        .capabilities = MTOUCH_CAPABILITIES_CONTACT_ID |
                        MTOUCH_CAPABILITIES_COORDS,
        .flags = 0,
        .max_touchpoints = p_dev->max_touchpoints,
        .width = p_dev->width,
        .height= p_dev->height,
    };

    if (params.max_touchpoints > SYNA_MAX_TOUCH_POINTS)
        params.max_touchpoints = SYNA_MAX_TOUCH_POINTS;

    LOGI("Touch points supported: %-2d", params.max_touchpoints);
    LOGI("Touch resolution supported: (%-4d , %-4d)", params.width, params.height);

    /* connect the driver with the specified parameters and callback functions
     * to the Input Events framework. It must be called while initializing the
     * touch screen driver.
     */
    p_dev->inputevents_hdlr = mtouch_driver_attach(&params, &funcs);

    if (NULL == p_dev->inputevents_hdlr) {
        LOGE("Failed to connect to libinputevents");
        error_memory("Synaptics_Touch: Failed to connect to libinputevents");
        return -EIO;
    }

    return EOK;
}
/**
 * mtouch_options()
 *
 * Parse the argument options from the graphics.conf.
 *
 * This is a callback function after calling the input_parseopts()
 *
 * The input options is formatted as below:
 *        option=value1,option2=value2
 *
 * @param
 *    [ in] option: passed from mtouch section of graphics.conf
 *    [ in] value:  value for each arguments
 *    [ in] arg:    user information
 *
 * @return
 *    0 on success; otherwise, on error.
 */
syna_static int mtouch_options(const char *option, const char *value, void *arg)
{
    int retval;
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)arg;
    struct syna_hw_bus_data *bus;

    _CHECK_POINTER(p_dev);

    bus = &p_dev->hw_if->bdata_io;

    if (0 == strcmp("i2c_dev_name", option)) {
        bus->type = BUS_TYPE_I2C;
        retval = input_parse_string(option, value, &bus->i2c_path);
        if(p_dev->verbose > 5)
            LOGD("i2c device: %s", bus->i2c_path);
        return retval;
    }
    else if (0 == strcmp("slave_addr", option)) {
        retval = input_parse_unsigned(option, value, &bus->i2c_addr);
        if(p_dev->verbose > 5)
            LOGD("i2c slave address: 0x%2X", bus->i2c_addr);
        return retval;
    }
    else if (0 == strcmp("i2c_speed", option)) {
        retval = input_parse_unsigned(option, value, &bus->frequency_hz);
        // if(p_dev->verbose > 5)
            LOGD("bus frequency: %d", bus->frequency_hz);
        return retval;
    }
    else if (0 == strcmp("spi_devname", option)) {
        bus->type = BUS_TYPE_SPI;
        retval = input_parse_string(option, value, &bus->spi_path);
        if(p_dev->verbose > 5)
            LOGD("spi device: %s", bus->spi_path);
        return retval;
    }
    else if (0 == strcmp("spi_mode", option)) {
        retval = input_parse_unsigned(option, value, &bus->spi_mode);
        if(p_dev->verbose > 5)
            LOGD("spi mode: %d", bus->spi_mode);
        return retval;
    }
    else if (0 == strcmp("bus_freq", option)) {
        retval = input_parse_unsigned(option, value, &bus->frequency_hz);
        if(p_dev->verbose > 5)
            LOGD("bus frequency: %d", bus->frequency_hz);
        return retval;
    }
    else if (0 == strcmp("fw_update_startup", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->startup_fw_update);
        if(p_dev->verbose > 5)
            LOGD("do fw update: %d", p_dev->startup_fw_update);
        return retval;
    }
    else if (0 == strcmp("fw_img", option)) {
        retval = input_parse_string(option, value, &p_dev->fw_image_path);
        if(p_dev->verbose > 5)
            LOGD("firmware image given: %s", bus->i2c_path);
        return retval;
    }
    else if (0 == strcmp("win_height", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->win_height);
        if(p_dev->verbose > 5)
            LOGD("qvm window height given: %d", p_dev->win_height);
        return retval;
    }
    else if (0 == strcmp("win_width", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->win_width);
        if(p_dev->verbose > 5)
            LOGD("qvm window width given: %d", p_dev->win_width);
        return retval;
    }
    else if (0 == strcmp("win_xpos", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->win_xpos);
        if(p_dev->verbose > 5)
            LOGD("qvm window x pos given: %d", p_dev->win_xpos);
        return retval;
    }
    else if (0 == strcmp("win_ypos", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->win_ypos);
        if(p_dev->verbose > 5)
            LOGD("qvm window y pos given: %d", p_dev->win_ypos);
        return retval;
    }
    else if (0 == strcmp("disp_id", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->disp_id);
        if(p_dev->verbose > 5)
            LOGD("qvm window display id given: %d", p_dev->disp_id);
        return retval;
    }
    else if (0 == strcmp("intr_gpio_pin", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->hw_if->bdata_attn.irq_gpio);
        if(p_dev->verbose > 5)
            LOGD("gpio num given: %d", p_dev->hw_if->bdata_attn.irq_gpio);
        return retval;
    }
    else if (0 == strcmp("pm_dev", option)) {
        retval = input_parse_string(option, value, &p_dev->pm_dev);
        if(p_dev->verbose > 5)
            LOGD("pm dev: %s", p_dev->pm_dev);
        return retval;
    }
    else if (0 == strcmp("verbose", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->verbose);
        if(p_dev->verbose > 5)
            LOGD("verbose level: %d", p_dev->verbose);
        return retval;
    }
    else if (0 == strcmp("fidm_attach_point", option)) {
        retval = input_parse_string(option, value, &p_dev->msg_thread->fidm_attach_point);
        if(p_dev->verbose > 5)
            LOGD("fidm attach point: %s", p_dev->msg_thread->fidm_attach_point);
        return retval;
    }
    else if (0 == strcmp("win_name", option)) {
        retval = input_parse_string(option, value, &p_dev->win_name);
        if(p_dev->verbose > 5)
            LOGD("fidm attach point: %s", p_dev->win_name);
        return retval;
    }
    else if (0 == strcmp("dev_ctrl_path", option)) {
        retval = input_parse_string(option, value, &p_dev->hw_if->bdata_rst.dev_ctrl_path);
        if(p_dev->verbose > 5)
            LOGD("fidm attach point: %s", p_dev->hw_if->bdata_rst.dev_ctrl_path);
        return retval;
    }
    else if (0 == strcmp("dev_status_path", option)) {
        retval = input_parse_string(option, value, &p_dev->dev_status_path);
        if(p_dev->verbose > 5)
            LOGD("fidm attach point: %s", p_dev->dev_status_path);
        return retval;
    }
    else if (0 == strcmp("sync_point", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->sync_point);
        if(p_dev->verbose > 5)
            LOGD("Sync point: %d", p_dev->sync_point);
        return retval;
    }
    else  if (0 == strcmp("width", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->width);
        if(p_dev->verbose > 5)
            LOGD("Touch resolution width: %d", p_dev->width);
        return retval;
    }
    else if (0 == strcmp("height", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->height);
        if(p_dev->verbose > 5)
            LOGD("Touch resolution height: %d", p_dev->height);
        return retval;
    }
    else if (0 == strcmp("num_touchpts", option)) {
        retval = input_parse_unsigned(option, value, &p_dev->max_touchpoints);
        if(p_dev->verbose > 5)
            LOGD("Max Touch points: %d", p_dev->max_touchpoints);
        return retval;
    }
    else {
        LOGE("Invalid option: '%s'", option);
    }

    return EOK;
}


/**
 * mtouch_tcm_set_up_app_fw()
 *
 * Implement the essential steps for the initialization including the
 * preparation of app info and the configuration of touch report.
 *
 * This function should be called whenever the device initially powers
 * up, resets, or firmware update.
 *
 * @param
 *    [ in] tcm: tcm driver handle
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
syna_static int mtouch_tcm_set_up_app_fw(struct syna_tcm_dev *p_dev)
{
    int retval = 0;
    struct tcm_dev *tcm_dev;

    _CHECK_POINTER(p_dev);

    tcm_dev = p_dev->tcm_dev;

    if (IS_NOT_APP_FW_MODE(tcm_dev->dev_mode)) {
        LOGN("Application firmware not running, current mode: %02x\n",
            tcm_dev->dev_mode);
        return -EINVAL;
    }

    /* collect app info containing most of sensor information */
    retval = syna_tcm_get_app_info(tcm_dev, &tcm_dev->app_info);
    if (retval < 0) {
        LOGE("Fail to get application info\n");
        error_memory("Synaptics_Touch: Fail to get application info\n");
        return _EIO;
    }

    /* set up the format of touch report */
#ifdef USE_CUSTOM_TOUCH_REPORT_CONFIG
    retval = syna_tcm_set_touch_report_config(tcm_dev,
            custom_touch_format,
            (unsigned int)sizeof(custom_touch_format));
    if (retval < 0) {
        LOGE("Fail to setup the custom touch report format\n");
        error_memory("Synaptics_Touch: Fail to get application info\n");
        return _EIO;
    }
#endif
    /* preserve the format of touch report */
    retval = syna_tcm_preserve_touch_report_config(tcm_dev);
    if (retval < 0) {
        LOGE("Fail to preserve touch report config\n");
        error_memory("Synaptics_Touch: Fail to preserve touch report config\n");
        return _EIO;
    }

    return retval;
}

#ifdef STARTUP_REFLASH
/**
 * mtouch_tcm_fw_update()
 *
 * This function calls tcm core library to perform the fw update
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
static int mtouch_tcm_fw_update(struct syna_tcm_dev *p_dev)
{
    int retval;
    int num_bytes;
    FILE *fp;
    unsigned char *bin_data;
    unsigned int bin_data_size;
    int erase_delay;
    int write_delay;
    unsigned int len_per_line;
    char line[IHEX_RECORD_SIZE + 1] = {0};
    enum REFLASH_TYPE type = REFLASH_TARGET_DUT;
    bool is_multichip;

    _CHECK_POINTER(p_dev);

    LOGI("Firmware file: %s", p_dev->fw_image_path);

    if (strlen(p_dev->fw_image_path) <= 0) {
        LOGE("Invalid path of image file");
        error_memory("Synaptics_Touch: Invalid path of image file");
        return _EINVAL;
    }

    /* read in binary data from the given path */
    fp = fopen(p_dev->fw_image_path, "rb");
    if (!fp) {
        LOGE("image file %s not found", p_dev->fw_image_path);
        error_memory("Synaptics_Touch: image file %s not found", p_dev->fw_image_path);
        return _EINVAL;
    }

    fseek(fp, 0L, SEEK_END);
    bin_data_size = ftell(fp);

    LOGI("Firmware file size = %d", bin_data_size);
    if (bin_data_size <= 0) {
        LOGE("Fail to request the size of %s", p_dev->fw_image_path);
        error_memory("Synaptics_Touch: Fail to request the size of %s", p_dev->fw_image_path);
        fclose(fp);
        return _EINVAL;
    }

    fseek(fp, 0L, SEEK_SET);

    bin_data = calloc((bin_data_size + 1), sizeof(unsigned char));
    if (!bin_data) {
        LOGE("Fail to allocate memory for binary buffer (size = %d)",
                bin_data_size + 1);

        return _ENOMEM;
    }

    if (IS_IHEX_REFLASH(type)) {
        /* data reordering from ihex file to the buffer */
        num_bytes = 0;
        len_per_line = 0;
        while (fgets(line, sizeof(line), fp) != NULL) {
            line[strcspn(line, "\r\n")] = 0;

            if (strlen(line) == 0)
                continue;

            if (len_per_line == 0)
                len_per_line = (unsigned int)strlen(line);

            if (num_bytes >= bin_data_size) {
                LOGE("No enough space to save ihex strings, offset:%d (max:%d)\n",
                        num_bytes, bin_data_size);
                error_memory("Synaptics_Touch: No enough space to save ihex strings, offset:%d (max:%d)\n", num_bytes, bin_data_size);
                free(bin_data);
                fclose(fp);
                return -EFBIG;
            }

            memcpy(&bin_data[num_bytes], line, len_per_line);
            num_bytes += len_per_line;
        }

        LOGI("Firmware num_bytes = %d, line_unit = %d", num_bytes, len_per_line);
    }
    else { /* data cloning from image file to the buffer */
        num_bytes = fread(bin_data,
                        sizeof(unsigned char),
                        bin_data_size,
                        fp);
        if (num_bytes != bin_data_size) {
            LOGE("Fail to get entire content of image file (bytes read = %d)(binary size = %d)",
                    num_bytes, bin_data_size);
            error_memory("Synaptics_Touch: Fail to get entire content of image file (bytes read = %d)(binary size = %d)", num_bytes, bin_data_size);
            free(bin_data);
            fclose(fp);
            return -EFBIG;
        }

        LOGI("Firmware num_bytes = %d,", num_bytes);
    }

    fclose(fp);

    is_multichip = IS_REFLASH_TO_MULTICHIP_TDDI(type);

    erase_delay = FLASH_ERASE_POLLING_DELAY;
    write_delay = FLASH_WRITE_POLLING_DELAY;

    if (IS_IHEX_REFLASH(type)) {
        /* perform ihex update for tddi device */
        retval = syna_tcm_romboot_do_ihex_update(p_dev->tcm_dev,
                bin_data,
                num_bytes,
                num_bytes + 4096,
                len_per_line,
                (erase_delay << 16) | (write_delay),
                is_multichip);
    }
    else if (type == IMAGE_REFLASH_MULTIPLECHIP_TDDI) {
        /* perform fw update for multi-chip tddi device */
        retval = syna_tcm_romboot_do_multichip_reflash(p_dev->tcm_dev,
                bin_data,
                bin_data_size,
                (erase_delay << 16) | (write_delay),
                false);
    }
    else {
        /* perform fw update for legacy touch device as well as
         * single chip TDDI product
         */
        retval = syna_tcm_do_fw_update(p_dev->tcm_dev,
                bin_data,
                bin_data_size,
                (erase_delay << 16) | (write_delay),
                false);
    }

    LOGN("Reflash %s\n", (retval < 0) ? "failure" : "done");

    if (bin_data)
        free(bin_data);

    return retval;
}
#endif  /* End of STARTUP_REFLASH */

/**
 * mtouch_tcm_disconnect()
 *
 * This function will power off the connected device.
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
static int mtouch_tcm_disconnect(struct syna_tcm_dev *p_dev)
{
    struct syna_hw_interface *hw_if;
    struct tcm_dev *tcm_dev;

    _CHECK_POINTER(p_dev);

    hw_if = p_dev->hw_if;
    tcm_dev = p_dev->tcm_dev;

    if (!tcm_dev) {
        LOGE("Invalid tcm_dev\n");
        error_memory("Synaptics_Touch: Invalid tcm_dev\n");
        return -EINVAL;
    }

    if (!hw_if) {
        LOGE("Invalid hardware interface\n");
        error_memory("Synaptics_Touch: Invalid hardware interface\n");
        return _EINVAL;
    }
    if (p_dev->is_connected == false) {
        LOGI("%s already disconnected\n", PLATFORM_DRIVER_NAME);
        return 0;
    }

    /* power off */
    if (hw_if->ops_power_on)
        hw_if->ops_power_on(hw_if, false);

    p_dev->is_connected = false;

    LOGI("%s device disconnected\n", PLATFORM_DRIVER_NAME);

    return 0;
}
/**
 * mtouch_tcm_connect()
 *
 * This function will power on and identify the connected touchcomm
 * device.
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
syna_static int mtouch_tcm_connect(struct syna_tcm_dev *p_dev)
{
    int retval;
    struct syna_hw_interface *hw_if;
    struct tcm_dev *tcm_dev;

    _CHECK_POINTER(p_dev);

    hw_if = p_dev->hw_if;

    tcm_dev = p_dev->tcm_dev;


    if (!tcm_dev) {
        LOGE("Invalid tcm_dev\n");
        error_memory("Synaptics_Touch: Invalid tcm_dev\n");
        return _EINVAL;
    }

    if (!hw_if) {
        LOGE("Invalid hardware interface\n");
        error_memory("Synaptics_Touch: Invalid hardware interface\n");
        return _EINVAL;
    }

    if (p_dev->is_connected == true) {
        LOGI("%s already connected\n", PLATFORM_DRIVER_NAME);
        return 0;
    }

    /* power on the connected device */
    if (hw_if->ops_power_on) {
        retval = hw_if->ops_power_on(hw_if, true);
        if (retval < 0)
            return _ENODEV;
    }

    /* perform a hardware reset */
    if (hw_if->ops_hw_reset) {
        retval = hw_if->ops_hw_reset(hw_if);
        if (retval < 0) {
            LOGE("Failed to execute hard reset of touch controller %d\n", retval);
            error_memory("Synaptics_Touch: Failed to execute hard reset of touch controller %d\n", retval);
            return _ENODEV;
        }
        LOGI("Executed hard resest successfully\n");
    }
    syna_pal_sleep_ms(500);

    /* detect which modes of touch controller is running */
    retval = syna_tcm_detect_device(tcm_dev);
    if (retval < 0) {
        LOGE("Fail to detect the device\n");
        error_memory("Synaptics_Touch: Fail to detect the device\n");
        return retval;
    }

    if (MODE_APPLICATION_FIRMWARE == tcm_dev->dev_mode) {
        retval = mtouch_tcm_set_up_app_fw(p_dev);
        if (retval < 0) {
            LOGE("Fail to set up application firmware\n");
            error_memory("Synaptics_Touch: Fail to set up application firmware\n");
            return retval;
        }
    }
    else if (IS_BOOTLOADER_MODE(tcm_dev->dev_mode) ||
            IS_ROM_BOOTLOADER_MODE(tcm_dev->dev_mode)) {
        LOGN("Application firmware not running, current mode: 0x%02x\n",
            retval);
    }
    else {
        LOGE("Unknown device mode, fail to connect to device\n");
        error_memory("Synaptics_Touch: Unknown device mode, fail to connect to device\n");
        return retval;
    }

    memset(p_dev->prev_obj_status, 0x00, sizeof(p_dev->prev_obj_status));

    p_dev->is_connected = true;

    LOGI("Connect to tcm device %s\n", PLATFORM_DRIVER_NAME);

    return 0;
}

/**
 * mtouch_create_isr()
 *
 * Set up the interrupt handler thread and a communication channel
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    on success, 0; otherwise, negative value on error.
 */
static int mtouch_create_isr(struct syna_tcm_dev *p_dev)
{
    int retval;

    _CHECK_POINTER(p_dev);

    p_dev->status = STATUS_INIT;

    /* setup the interrupt communication channel */
    p_dev->thread_chid = ChannelCreate(_NTO_CHF_DISCONNECT | _NTO_CHF_UNBLOCK);
    if (p_dev->thread_chid < 0) {
        LOGE("Fail to create interrupt handling channel (error: %s)",
            strerror(errno));
        error_memory("Synaptics_Touch: Fail to create interrupt handling channel (error: %s)", strerror(errno));
        return _EIO;
    }
    p_dev->thread_coid = ConnectAttach(0, 0, p_dev->thread_chid, _NTO_SIDE_CHANNEL, 0);
    if (p_dev->thread_coid < 0) {
        LOGE("Fail to attach interrupt handling channel (error: %s)",
            strerror(errno));
        error_memory("Synaptics_Touch: Fail to attach interrupt handling channel (error: %s)", strerror(errno));
        return _EIO;
    }

    /* initializes the thread attributes in defaults
     * defaults: PTHREAD_CANCEL_DEFERRED, PTHREAD_CANCEL_ENABLE, PTHREAD_CREATE_JOINABLE
     * PTHREAD_INHERIT_SCHED, PTHREAD_SCOPE_SYSTEM
     */
    pthread_attr_init(&p_dev->thread_attr);
    /* set the thread scheduling policy, round-robin scheduling. */
    pthread_attr_setschedpolicy(&p_dev->thread_attr, SCHED_RR);
    /* set a thread's scheduling parameters */
    p_dev->thread_param.sched_priority = THREAD_PRIORITY;
    pthread_attr_setschedparam(&p_dev->thread_attr, &p_dev->thread_param);
    /* set the thread's inherit-scheduling attribute
     * use the scheduling policy specified in pthread_attr_t for the thread.
     */
    pthread_attr_setinheritsched(&p_dev->thread_attr, PTHREAD_EXPLICIT_SCHED);
    /* create the thread in a detached state. */
    pthread_attr_setdetachstate(&p_dev->thread_attr, PTHREAD_CREATE_DETACHED);

    /* describe the signal event, send a pulse
     * sigev_coid: the connection ID. this should be attached to the channel with
     *             which the pulse will be received.
     * sigev_code: a code to be interpreted by the pulse handler
     */
    p_dev->thread_event.sigev_notify = SIGEV_PULSE;
    p_dev->thread_event.sigev_coid = p_dev->thread_coid;
    p_dev->thread_event.sigev_code = PULSE_CODE;

    /* create interrupt handler thread */
    retval = pthread_create(&p_dev->isr_thread,
                        &p_dev->thread_attr,
                        (void *)mtouch_interrupt_handling_thread,
                        p_dev);
    if (EOK != retval) {
        LOGE("Fail to do pthread_create (error: %s)", strerror(errno));
        error_memory("Synaptics_Touch: Fail to do pthread_create (error: %s)", strerror(errno));
        return _EIO;
    }
    /* name a thread
     * if a thread is setting its own name, uses ThreadCtl()
     */
    pthread_setname_np(p_dev->isr_thread, "mtouch-synaptics-isr");

    // attach the given event to an interrupt source
    // assigning the interrupt vector number and the pointer of sigvent structure
    // that want to be delivered when this interrupt occurs
    //
    // before calling InterruptAttachEvent, it must request I/O privileges by calling ThreadCtl()
    if (EOK != syna_mtouch_tp_intr_attach(p_dev)) {
        LOGE("Failed to attach to interrupt (error: %s)", strerror(errno));
        error_memory("Synaptics_Touch: Failed to attach to interrupt (error: %s)", strerror(errno));
        return _EIO;
    }

    LOGI("Interrupt handler registered\n");

    return 0;
}
/**
 * mtouch_driver_remove()
 *
 * Release all resources allocated previously
 *
 * @param
 *    [ in] p_dev: tcm device driver handle
 *
 * @return
 *    none
 */
static void mtouch_driver_remove(struct syna_tcm_dev *p_dev)
{
    if (!p_dev) {
        return;
    }
    /* disconnect the touchcomm driver */
    mtouch_tcm_disconnect(p_dev);

    /* destroy the isr thread */
    p_dev->status = STATUS_UNKNOWN;
    pthread_cancel(p_dev->isr_thread);
    pthread_join(p_dev->isr_thread, NULL);

    pthread_cancel(p_dev->msg_thread->fidm_thread);
    pthread_join(p_dev->msg_thread->fidm_thread, NULL);

    /* detach an interrupt handler */
    if (p_dev->hw_if->bdata_attn.irq_id >= 0) {
        InterruptDetach(p_dev->hw_if->bdata_attn.irq_id);
    }

    /* detach the mtouch driver */
    if (p_dev->inputevents_hdlr) {
        mtouch_driver_detach(p_dev->inputevents_hdlr);
        p_dev->inputevents_hdlr = NULL;
    }

    syna_pal_mutex_free(&p_dev->tp_event_mutex);

    /* remove the allocated tcm device */
    if (p_dev->tcm_dev) {
        syna_tcm_remove_device(p_dev->tcm_dev);
        p_dev->tcm_dev = NULL;
    }
    /* close spi interface */
    if (p_dev->hw_if->bdata_io.spi_path) {
        syna_pal_mem_free(p_dev->hw_if->bdata_io.spi_path);
        p_dev->hw_if->bdata_io.spi_path = NULL;
    }
    if (p_dev->hw_if->bdata_io.spi_fd) {
        // spi_close(p_dev->hw_if->bdata_io.spi_fd);
        p_dev->hw_if->bdata_io.spi_fd = -1;
    }
    /* close i2c interface */
    if (p_dev->hw_if->bdata_io.i2c_path) {
        syna_pal_mem_free(p_dev->hw_if->bdata_io.i2c_path);
        p_dev->hw_if->bdata_io.i2c_path = NULL;
    }
    if (p_dev->hw_if->bdata_io.i2c_fd) {
        i2c_close(p_dev->hw_if->bdata_io.i2c_fd);
        p_dev->hw_if->bdata_io.i2c_fd = -1;
    }

    syna_tcm_buf_release(&p_dev->event_data);

    if (p_dev->tp_data) {
        syna_pal_mem_free(p_dev->tp_data);
        p_dev->tp_data = NULL;
    }

    p_dev->faceplate_coid = -1;

    if (p_dev->thread_chid >= 0) {
        ChannelDestroy(p_dev->thread_chid);
        p_dev->thread_chid = -1;
        ConnectDetach(p_dev->thread_coid);
        p_dev->thread_coid = -1;
    }

    if (p_dev->msg_thread->fidm_attach) {
        ChannelDestroy(p_dev->msg_thread->fidm_attach->chid);
        p_dev->msg_thread->fidm_attach->chid = -1;
        ConnectDetach(p_dev->msg_thread->fidm_coid);
        p_dev->msg_thread->fidm_coid = -1;
        name_detach(p_dev->msg_thread->fidm_attach, 0);
        p_dev->msg_thread->fidm_attach = NULL;
    }

    /* release syna tcm device */
    syna_pal_mem_free(p_dev);
    p_dev = NULL;

    return;
}

int syna_mtouch_suspend(struct syna_tcm_dev *dev)
{
    int ret;
    struct syna_hw_interface *hw_if;

    hw_if = dev->hw_if;
    ret = mtouch_irq_enable(hw_if, 0);
    if (ret < 0) {
        LOGE("failed to disable interrupt");
        error_memory("Synaptics_Touch: failed to disable interrupt");
        return ret;
    }
    ret = syna_tcm_sleep(dev->tcm_dev, 1);
    if (ret < 0) {
        LOGE("failed to set device into sleep mode");
        error_memory("Synaptics_Touch: failed to set device into sleep mode");
    }
    syna_release_touch(dev);

    return 0;
}


int syna_mtouch_resume(struct syna_tcm_dev *dev)
{
    int ret;
    struct syna_hw_interface *hw_if;
    int retry_count = 0;

    dev->is_connected = false;

retry_init:
    ret = mtouch_tcm_connect(dev);
    if (ret < 0) {
      /*hardcode 10 retry attempts on RESUME*/
       if (retry_count < 10) {
          retry_count++;
          LOGI("HW initialization failed after suspend, retrying %d\n",retry_count);
          if (usleep(500*1000) != 0) {
             LOGE("Failed to execute 500ms sleep %d\n", errno);
             error_memory("Synaptics_Touch: Failed to execute 500ms sleep %d\n", errno);
          }
          goto retry_init;
       } else {
          LOGE("HW initialization failed after suspend %d", ret);
          error_memory("Synaptics_Touch: HW initialization failed after suspend %d", ret);
          return ret;
      }
    }

    hw_if = dev->hw_if;
    ret = mtouch_irq_enable(hw_if, 1);
    if (ret < 0) {
        LOGE("failed to enable interrupt");
        error_memory("Synaptics_Touch: failed to enable interrupt");
        return ret;
    }

    return 0;
}
/**
 * syna_mtouch_connect_faceplate()
 *
 * This function connects with faceplate resource manager.
 * The coid is used to send the knob data to faceplate rm.
 *
 * @param
 *    [ in] p_dev: pointer to the synaptics device handle.
 *
 * @return
 *    0 if the driver is connected to faceplate,
 *    else returns a negative error code if failed to connect
 */
static int syna_mtouch_connect_faceplate(struct syna_tcm_dev *p_dev)
{
    char *attach_point;
    attach_point = strdup(FACEPLATE_ATTACH_POINT);
    p_dev->faceplate_coid = name_open(attach_point, 0);
    if (p_dev->faceplate_coid == -1)
    {
        LOGE("Failed to connect to faceplate driver, coid: %d, %d", p_dev->faceplate_coid, errno);
        return -1;
    }
    LOGE("Connected to faceplate driver successfully");

    return 0;
}

/**
 * mtouch_driver_init()
 *
 * The initialization callback.
 *
 * QNX Screen calls dlopen() on this specified driver entry.
 * Upon a successful dlopen(), Screen will use dlsym() to look for mtouch_driver_init()
 * for the driver initialization.
 *
 * The followings are performed:
 *    - set the default values of the driver
 *    - parse any options defined in graphics.conf
 *    - hardware initialization
 *    - create a separate thread to interact with the hardware and trigger the input events
 *      library API function mtouch_driver_process_packet()
 *    - initialize the touch function
 *    - connect to input events library
 *
 * @param
 *    [ in] options: settings defined in graphics.conf
 *
 * @return
 *    0 if the driver registered and bound to a device,
 *    else returns a negative error code and with the driver not registered.
 */
void *mtouch_driver_init(const char *options)
{
    int retval;
    struct syna_tcm_dev *p_dev = NULL;
    struct tcm_dev *tcm_dev = NULL;
    int retry_count = 0;
    int fd=0;
    struct syna_hw_interface *hw_if;

    LOGI("Start to initialize mtouch driver");

    if (InstallLocalErrMemReader(RINGBUFFER_SIZE) != 0)
    {
        LOGE("errmem low priority worker thread creation failed !!!");
    }

    /* create a syna tcm instance */
    p_dev = syna_pal_mem_alloc(1, sizeof(struct syna_tcm_dev));
    if (!p_dev) {
        LOGE("Fail to allocate syna tcm device ");
        error_memory("Synaptics_Touch: Fail to allocate syna tcm device ");
        return NULL;
    }

    p_dev->msg_thread = calloc (1, sizeof(*p_dev->msg_thread));

    /* initialize defaults */
    p_dev->tcm_dev = NULL;
    p_dev->status = STATUS_UNKNOWN;
    p_dev->is_connected = false;
    p_dev->inputevents_hdlr = NULL;
    p_dev->isr_thread = 0;
    p_dev->height = 720;
    p_dev->width = 1280;
    p_dev->win_height = 720;
    p_dev->win_width = 1280;
    p_dev->win_xpos = 0;
    p_dev->win_ypos = 0;
    p_dev->disp_id = 2;
    p_dev->pm_dev = PM_DEV_NODE;
    p_dev->msg_thread->fidm_attach_point = NULL;
    p_dev->dev_status_path = "/dev/vcd/display-binder/status";
    p_dev->win_name = "qvm_window";
    p_dev->display_group_id = -1;
    p_dev->sync_point = 1;
    p_dev->max_touchpoints = 5;

    syna_tcm_buf_init(&p_dev->event_data);

    p_dev->tp_data = syna_pal_mem_alloc(1, sizeof(struct tcm_touch_data_blob));
    if (!p_dev->tp_data) {
        LOGE("Fail to allocate touch data blob ");
        error_memory("Synaptics_Touch: Fail to allocate touch data blob ");
        syna_tcm_buf_release(&p_dev->event_data);
        if (p_dev) {
            syna_pal_mem_free(p_dev);
            p_dev = NULL;
        }
        return NULL;
    }

    p_dev->thread_chid = -1;
    p_dev->thread_coid = -1;
    p_dev->faceplate_coid = -1;
    p_dev->thread_param.sched_priority = THREAD_PRIORITY;
    p_dev->thread_event.sigev_priority = THREAD_PRIORITY;

    /* get the hardware specific interface */
    p_dev->hw_if = &mtouch_hw_if;

    /* initialize defaults at the hardware specific interface */
    p_dev->hw_if->bdata_attn.irq_gpio = GPIO_ATTN;
    p_dev->hw_if->bdata_attn.irq_on_state = 0; /* low-level triggered */

     p_dev->hw_if->bdata_rst.dev_ctrl_path = "/dev/vcd/display-binder/control";

    mtouch_bus_init(p_dev->hw_if);

    p_dev->fw_image_path = (char *)syna_pal_mem_alloc(512, sizeof(char));
    if (!p_dev->fw_image_path) {
        LOGE("Fail to create the fw_image string buffer");
        error_memory("Synaptics_Touch: Fail to create the fw_image string buffer");
        mtouch_driver_remove(p_dev);
        return NULL;
    }

    /* enable IO capability
     * lock the process's memory and request I/O privileges
     * let the thread execute the in, ins, out, outs, cli, and sti I/O opcodes
     */
    if (ThreadCtl(_NTO_TCTL_IO, 0) == -1) {
        LOGE("Fail to config ThreadCtl");
        error_memory("Synaptics_Touch: Fail to config ThreadCtl");
        mtouch_driver_remove(p_dev);
        return NULL;
    }

    /* parses settings defined in graphics.conf */
    input_parseopts(options, mtouch_options, p_dev);

    /* configure the hardware bus interface */
    retval = mtouch_bus_setup(p_dev->hw_if,
        p_dev->hw_if->bdata_io.type);

    /* allocate the TouchCom device handle */
    retval = syna_tcm_allocate_device(&tcm_dev, p_dev->hw_if);
    if ((retval < 0) || (!tcm_dev)) {
        mtouch_driver_remove(p_dev);
        return NULL;
    }

    p_dev->tcm_dev = tcm_dev;

    syna_pal_mutex_alloc(&p_dev->tp_event_mutex);

    /* connect to touchcomm device, and also complete the
     * basic initialization.
     *
     * at the end of function returned, the device should
     * stay at the application fw mode.
     */
retry_init:
    retval = mtouch_tcm_connect(p_dev);
    if (retval < 0) {
       if (retry_count < MAX_HW_RETRY_COUNT) {
          retry_count++;
          LOGI("HW initialization failed, retrying %d\n",retry_count);
          if (usleep(500*1000) != 0) {
             LOGE("Failed to execute 500ms sleep %d\n", errno);
             error_memory("Synaptics_Touch: Failed to execute 500ms sleep %d\n", errno);
          }
          goto retry_init;
       } else {
          LOGE("HW initialization failed %d", retval);
          error_memory("Synaptics_Touch: HW initialization failed, move to suspend %d", retval);
          /*HW init is failed, update pm status to suspend. Diag requests through the I2C bus will
          *be ignored
          */
          p_dev->syn_pm_state = PM_STATE_SUSPEND;

      }
    }
    else 
    {
        LOGI("HW initialization is sucesss  %d\n",retval);
        p_dev->syn_pm_state = PM_STATE_RESUME;
    }

#ifdef STARTUP_REFLASH
    /* call firmware updating if it is necessary
     * the control switch is defined in option "fw_update_startup" in graphics.conf
     */
    if (p_dev->startup_fw_update) {
        retval = mtouch_tcm_fw_update(p_dev);
        if (retval < 0) {
            LOGE("Fail to do fw update");
            error_memory("Synaptics_Touch: Fail to do fw update");
            mtouch_driver_remove(p_dev);
            return NULL;
        }

        retval = mtouch_tcm_set_up_app_fw(p_dev);
        if (retval < 0) {
            LOGE("Fail to set up fw after fw update");
            error_memory("Synaptics_Touch: Fail to set up fw after fw update");
            mtouch_driver_remove(p_dev);
            return NULL;
        }
    }
#endif

    /* create interrupt handler thread */
    retval = mtouch_create_isr(p_dev);
    if (retval < 0) {
        LOGE("Fail to create ISR thread");
        error_memory("Synaptics_Touch: Fail to create ISR thread");
        mtouch_driver_remove(p_dev);
        return NULL;
    }

    if (p_dev->syn_pm_state == PM_STATE_SUSPEND) {
        hw_if = p_dev->hw_if;
        mtouch_irq_enable(hw_if, 0);
    }

    if (p_dev->msg_thread->fidm_attach_point == NULL) {
        p_dev->msg_thread->fidm_attach_point = strdup (FIDM_TOUCH_ATTACH_POINT);
    }

    if (EOK != syna_mtouch_register_pm(p_dev))
    {
        LOGE("registration with %s failed",p_dev->pm_dev );
        error_memory("Synaptics_Touch: registration with %s failed",p_dev->pm_dev );
    }
    else
        LOGI("registration with %s success",p_dev->pm_dev);
  
     /* attach to input events framework, libinputevents, independent of HW status
      * set up mtouch_driver_funcs_t and mtouch_driver_params_t as well
      */
      retval = mtouch_attach_device(p_dev);
      if (EOK != retval) {
            LOGE("Fail attach the driver into input event library %d",retval );
            error_memory("Synaptics_Touch: Fail attach the driver into input event library");
            mtouch_driver_remove(p_dev);
            return NULL;
       }

           /* create a window to forward events to guest OS */
    if (0 > mtouch_create_qvm_window(p_dev)) {
        LOGE("Failed to create window for qvm\n");
        error_memory("Synaptics_Touch: Failed to create window for qvm\n");
        if (p_dev) {
            syna_pal_mem_free(p_dev);
            p_dev = NULL;
        }
        return NULL;
    }

    /*Connect to faceplate driver to send knob data*/
    retval = syna_mtouch_connect_faceplate(p_dev);
    if (retval != 0) {
        LOGE("Failed to connect to faceplate driver during init, try again %d", retval);
    }

    /* Create channel for external messages */
    if ((p_dev->msg_thread->fidm_attach = name_attach(NULL, p_dev->msg_thread->fidm_attach_point, 0)) == NULL) {
        LOGE("%s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
        error_memory("Synaptics_Touch: %s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
        goto fail;
    }
    if (EOK != pthread_create(&p_dev->msg_thread->fidm_thread, NULL, syna_ext_msg_handler, p_dev)) {
        LOGE("Failed to create the external message thread");
        error_memory("Synaptics_Touch: Failed to create the external message thread");
        name_detach(p_dev->msg_thread->fidm_attach, 0);
        p_dev->msg_thread->fidm_attach = NULL;
        goto fail;
    }
    p_dev->msg_thread->fidm_coid = ConnectAttach(0, 0, p_dev->msg_thread->fidm_attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == p_dev->msg_thread->fidm_coid) {
        LOGE("%s: Message handler thread ConnectAttach: %s", __FUNCTION__, strerror(errno));
        error_memory("Synaptics_Touch: %s: Message handler thread ConnectAttach: %s", __FUNCTION__, strerror(errno));
        pthread_cancel(p_dev->msg_thread->fidm_thread);
        pthread_join(p_dev->msg_thread->fidm_thread, NULL);
        ChannelDestroy(p_dev->msg_thread->fidm_attach->chid);
        p_dev->msg_thread->fidm_attach->chid = -1;
        name_detach(p_dev->msg_thread->fidm_attach, 0);
        p_dev->msg_thread->fidm_attach = NULL;
    }

    /* Display Group ID */
    retval = syna_tcm_get_display_grpid(p_dev);
    if(retval != 0)
    {
        LOGE("Failed to fetch Display Group Id, try again %d", retval);
        error_memory("Synaptics_Touch: Failed to fetch Display Group Id, try again %d", retval);
    }

fail:
    LOGI("TouchComm mtouch driver, %s, v%d.%02d.%02d installed\n",
        PLATFORM_DRIVER_NAME,
        SYNAPTICS_TCM_QNX_DRIVER_VERSION_MAJOR,
        SYNAPTICS_TCM_QNX_DRIVER_VERSION_MINOR,
        SYNAPTICS_TCM_QNX_DRIVER_VERSION_MINORS);
    LOGI("Synaptics touch driver initialized successfully");

    /* Create file to signal that the driver is initialized and ready to serve clients */
    if (p_dev->sync_point) {
        fd=open("/tmp/sync/mtouch_initialized", O_WRONLY | O_CREAT | O_TRUNC,
        S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
      if (fd==-1) {
          LOGE("%s failed to create/open sync file :%s", __FUNCTION__,strerror(errno));
          error_memory("Synaptics_Touch: %s failed to create sync file", __FUNCTION__);
      } else {
          close(fd);
      }
    } else {
        LOGI("diagnostics sync point not enabled %d", p_dev->sync_point);
    }

    return p_dev;
}

/**
 * mtouch_driver_fini()
 *
 * The cleanup callback.
 *
 * mtouch_driver_fini() performs any necessary device cleanup
 * and disconnects from the Input Events library.
 *
 * @param
 *    [ in] dev: the device allocated previously
 *
 * @return
 *    0 if the driver registered and bound to a device,
 */
void mtouch_driver_fini(void *dev)
{
    struct syna_tcm_dev *p_dev = (struct syna_tcm_dev *)dev;

    mtouch_driver_remove(p_dev);
}

